<?php
/**
 * Covel Veículos - Premium Catalog v5
 * Professional design with calculated proportions
 * A4 = 842px height → 3 sections of 280px each
 */

ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', 'debug.log');
error_reporting(E_ALL);

// CORS
if (isset($_SERVER['HTTP_ORIGIN'])) {
    header("Access-Control-Allow-Origin: {$_SERVER['HTTP_ORIGIN']}");
    header('Access-Control-Allow-Credentials: true');
}

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_METHOD']))
        header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
    if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS']))
        header("Access-Control-Allow-Headers: {$_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS']}");
    header("Access-Control-Allow-Private-Network: true");
    exit(0);
}

header("Content-Type: application/pdf");
header("Content-Disposition: attachment; filename=catalogo-covel.pdf");

require 'vendor/autoload.php';
require 'ai_service.php';

use Dompdf\Dompdf;
use Dompdf\Options;

// Input
$inputJSON = file_get_contents('php://input');
$input = json_decode($inputJSON, true);

if (!$input || !isset($input['id'])) {
    http_response_code(400);
    die('Vehicle ID required');
}

$vehicleId = $input['id'];
$userName = $input['user_name'] ?? 'Cliente';

// Fetch XML
$xmlUrl = 'https://app.revendamais.com.br/application/index.php/apiGeneratorXml/generator/sitedaloja/5358d2c5b178fd7a21b077dcd41fa3f65199.xml';
$context = stream_context_create(["http" => ["header" => "User-Agent: CovelPDF/5.0"]]);
libxml_set_streams_context($context);
$xml = simplexml_load_file($xmlUrl, 'SimpleXMLElement', LIBXML_NOCDATA);

if (!$xml) die('XML failed');

// Find Vehicle
$vehicle = null;
foreach ($xml->AD as $ad) {
    if (trim((string)$ad->ID) === trim($vehicleId)) {
        $vehicle = $ad;
        break;
    }
}
if (!$vehicle) die('Not found');

// Images
$images = [];
if (isset($vehicle->IMAGES_LARGE->IMAGE_URL_LARGE)) {
    foreach($vehicle->IMAGES_LARGE->IMAGE_URL_LARGE as $img) $images[] = (string)$img;
} elseif (isset($vehicle->IMAGES->IMAGE_URL)) {
    foreach($vehicle->IMAGES->IMAGE_URL as $img) $images[] = (string)$img;
}
$img1 = $images[0] ?? 'https://via.placeholder.com/800x400/111/333?text=.';
$img2 = $images[1] ?? $img1;
$img3 = $images[2] ?? $img1;

// Vehicle Data
$v = [
    'model' => (string)$vehicle->TITLE ?: ((string)$vehicle->MAKE . ' ' . (string)$vehicle->MODEL),
    'price' => 'R$ ' . number_format((float)$vehicle->PRICE, 2, ',', '.'),
    'year' => (string)$vehicle->FABRIC_YEAR . '/' . (string)$vehicle->YEAR,
    'km' => number_format((float)$vehicle->MILEAGE, 0, ',', '.') . ' km',
    'fuel' => (string)$vehicle->FUEL,
    'gear' => (string)$vehicle->GEAR,
    'color' => (string)$vehicle->COLOR,
    'doors' => (string)$vehicle->DOORS,
    'accessories' => (string)$vehicle->ACCESSORIES
];

// AI Content
$ai = AIService::generateCatalogContent($v);

// Logo
$logoPath = __DIR__ . '/img/logo_covel.webp';
$logoSrc = '';
if (file_exists($logoPath) && function_exists('imagecreatefromwebp')) {
    $im = @imagecreatefromwebp($logoPath);
    if ($im) {
        ob_start();
        imagepng($im);
        $logoSrc = 'data:image/png;base64,' . base64_encode(ob_get_clean());
        imagedestroy($im);
    }
}

// Accessories (12 max)
$accList = array_filter(array_map('trim', explode(',', $v['accessories'])));
$accHtml = '';
foreach (array_slice($accList, 0, 12) as $a) {
    $accHtml .= "<span class='chip'>$a</span>";
}

// Highlights
$hlHtml = '';
$icons = ['◆', '◆', '◆', '◆'];
foreach ($ai['highlights'] as $i => $h) {
    $hlHtml .= "<div class='hl-row'><span class='hl-dot'>{$icons[$i]}</span><span class='hl-text'>$h</span></div>";
}

// ==================== HTML ====================
$html = <<<HTML
<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<style>
/* ===== RESET ===== */
@page { margin: 0; }
* { margin: 0; padding: 0; box-sizing: border-box; }

body { 
    font-family: Helvetica, Arial, sans-serif; 
    font-size: 10px; 
    color: #1a1a1a;
    line-height: 1.4;
    -webkit-font-smoothing: antialiased;
}

/* ===== PAGE STRUCTURE ===== */
/* A4 = 842px height. 3 sections = 280px each + 2px margin */
.page { 
    width: 100%; 
    height: 842px;
    page-break-after: always;
    position: relative;
    overflow: hidden;
}
.page:last-child { page-break-after: auto; }

/* Section = 1/3 of page height */
.sec {
    width: 100%;
    height: 280px;
    position: relative;
    overflow: hidden;
}

/* ===== SECTION BACKGROUNDS ===== */
.sec-img {
    position: absolute;
    top: 0; left: 0; right: 0; bottom: 0;
}
.sec-img img {
    width: 100%;
    height: 280px;
    object-fit: cover;
}
.sec-overlay {
    position: absolute;
    top: 0; left: 0; right: 0; bottom: 0;
    background: #000;
    opacity: 0.75;
}
.sec-content {
    position: relative;
    z-index: 5;
    height: 100%;
    padding: 25px 40px;
}

.bg-white { background: #fff; }
.bg-off { background: #fafbfc; }
.bg-light { background: #f0f2f5; }
.bg-dark { background: #0f0f0f; color: #fff; }
.bg-navy { background: #0a1628; color: #fff; }
.bg-blue { background: #0044cc; color: #fff; }

/* ===== TYPOGRAPHY ===== */
.f-8 { font-size: 8px; }
.f-9 { font-size: 9px; }
.f-10 { font-size: 10px; }
.f-11 { font-size: 11px; }
.f-12 { font-size: 12px; }
.f-14 { font-size: 14px; }
.f-16 { font-size: 16px; }
.f-18 { font-size: 18px; }
.f-20 { font-size: 20px; }
.f-24 { font-size: 24px; }
.f-28 { font-size: 28px; }

.fw-light { font-weight: 300; }
.fw-normal { font-weight: 400; }
.fw-medium { font-weight: 500; }
.fw-bold { font-weight: 700; }
.fw-black { font-weight: 900; }

.ls-1 { letter-spacing: 1px; }
.ls-2 { letter-spacing: 2px; }
.ls-3 { letter-spacing: 3px; }

.lh-1 { line-height: 1.0; }
.lh-12 { line-height: 1.2; }
.lh-14 { line-height: 1.4; }
.lh-16 { line-height: 1.6; }

.upper { text-transform: uppercase; }
.center { text-align: center; }
.justify { text-align: justify; }

.white { color: #fff; }
.gray { color: #666; }
.muted { opacity: 0.6; }
.muted-2 { opacity: 0.4; }

/* ===== SPACING ===== */
.mb-5 { margin-bottom: 5px; }
.mb-8 { margin-bottom: 8px; }
.mb-10 { margin-bottom: 10px; }
.mb-12 { margin-bottom: 12px; }
.mb-15 { margin-bottom: 15px; }
.mb-20 { margin-bottom: 20px; }
.mb-25 { margin-bottom: 25px; }

/* ===== COMPONENTS ===== */
.tag {
    display: inline-block;
    padding: 4px 12px;
    font-size: 7px;
    font-weight: 700;
    text-transform: uppercase;
    letter-spacing: 2px;
    border-radius: 3px;
    background: rgba(255,255,255,0.15);
    color: #fff;
}
.tag-solid {
    background: #0044cc;
    color: #fff;
}
.tag-accent {
    background: linear-gradient(135deg, #6366f1, #8b5cf6);
    color: #fff;
}

.price {
    display: inline-block;
    background: #0044cc;
    color: #fff;
    padding: 10px 28px;
    font-size: 18px;
    font-weight: 800;
    border-radius: 4px;
    letter-spacing: -0.5px;
}

.divider {
    width: 40px;
    height: 3px;
    background: #0044cc;
    margin-bottom: 15px;
}
.divider-white {
    background: rgba(255,255,255,0.3);
}

.logo { height: 28px; }
.logo-sm { height: 20px; }

/* ===== SPECS GRID ===== */
.specs {
    width: 100%;
    border-collapse: separate;
    border-spacing: 6px;
}
.specs td {
    background: #fff;
    padding: 12px 8px;
    text-align: center;
    border-radius: 4px;
    border: 1px solid #e8e8e8;
    width: 16.66%;
}
.spec-icon { font-size: 14px; margin-bottom: 4px; }
.spec-label { font-size: 7px; color: #999; text-transform: uppercase; letter-spacing: 1px; }
.spec-val { font-size: 10px; font-weight: 700; color: #1a1a1a; margin-top: 2px; }

/* ===== HIGHLIGHTS ===== */
.hl-row {
    display: block;
    padding: 8px 0;
    border-bottom: 1px solid rgba(255,255,255,0.1);
}
.hl-row:last-child { border-bottom: none; }
.hl-dot {
    color: #00c6ff;
    font-size: 8px;
    margin-right: 10px;
}
.hl-text {
    font-size: 10px;
    color: rgba(255,255,255,0.9);
}

/* ===== CHIPS ===== */
.chip {
    display: inline-block;
    background: #e8eaed;
    padding: 4px 10px;
    margin: 2px;
    border-radius: 3px;
    font-size: 8px;
    color: #444;
}

/* ===== MESSAGE BOX ===== */
.msg-card {
    background: rgba(255,255,255,0.08);
    border: 1px solid rgba(255,255,255,0.15);
    border-radius: 6px;
    padding: 18px;
}

/* ===== FOOTER BAR ===== */
.footer-bar {
    position: absolute;
    bottom: 0;
    left: 0;
    right: 0;
    height: 3px;
    background: linear-gradient(90deg, #0044cc, #00c6ff);
}

</style>
</head>
<body>

<!-- ====================== PAGE 1 ====================== -->
<div class="page">

    <!-- SEC 1: HERO -->
    <div class="sec">
        <div class="sec-img"><img src="{$img1}" /></div>
        <div class="sec-overlay"></div>
        <div class="sec-content center">
            <img src="{$logoSrc}" class="logo mb-15" />
            <div class="tag tag-accent mb-12">Seminovo Certificado</div>
            <div class="f-24 fw-black white lh-1 mb-8">{$ai['headline']}</div>
            <div class="f-10 white muted mb-15">{$ai['tagline']}</div>
            <div class="price">{$v['price']}</div>
        </div>
    </div>

    <!-- SEC 2: MODEL INFO -->
    <div class="sec bg-white">
        <div class="sec-content">
            <div class="divider"></div>
            <div class="f-16 fw-bold lh-12 mb-5">{$v['model']}</div>
            <div class="f-9 gray mb-15">{$v['year']} &nbsp;•&nbsp; {$v['km']} &nbsp;•&nbsp; {$v['fuel']} &nbsp;•&nbsp; {$v['gear']}</div>
            <div class="f-9 gray lh-16 justify">
                {$ai['description']}
            </div>
        </div>
    </div>

    <!-- SEC 3: SPECS -->
    <div class="sec bg-off">
        <div class="sec-content">
            <div class="f-8 upper ls-2 gray mb-8">Especificações Técnicas</div>
            <table class="specs">
                <tr>
                    <td>
                        <div class="spec-icon">📅</div>
                        <div class="spec-label">Ano</div>
                        <div class="spec-val">{$v['year']}</div>
                    </td>
                    <td>
                        <div class="spec-icon">🛣️</div>
                        <div class="spec-label">KM</div>
                        <div class="spec-val">{$v['km']}</div>
                    </td>
                    <td>
                        <div class="spec-icon">⛽</div>
                        <div class="spec-label">Combustível</div>
                        <div class="spec-val">{$v['fuel']}</div>
                    </td>
                    <td>
                        <div class="spec-icon">⚙️</div>
                        <div class="spec-label">Câmbio</div>
                        <div class="spec-val">{$v['gear']}</div>
                    </td>
                    <td>
                        <div class="spec-icon">🎨</div>
                        <div class="spec-label">Cor</div>
                        <div class="spec-val">{$v['color']}</div>
                    </td>
                    <td>
                        <div class="spec-icon">🚪</div>
                        <div class="spec-label">Portas</div>
                        <div class="spec-val">{$v['doors']}</div>
                    </td>
                </tr>
            </table>
            <div class="f-8 upper ls-2 gray mb-8" style="margin-top:18px;">Equipamentos de Série</div>
            <div>{$accHtml}</div>
        </div>
    </div>

    <div class="footer-bar"></div>
</div>

<!-- ====================== PAGE 2 ====================== -->
<div class="page">

    <!-- SEC 1: PERSONALIZED MESSAGE -->
    <div class="sec bg-navy">
        <div class="sec-content center">
            <div class="f-8 upper ls-3 muted mb-10">Catálogo Exclusivo Para</div>
            <div class="f-20 fw-bold white mb-15">{$userName}</div>
            <div class="msg-card" style="max-width:420px; margin:0 auto;">
                <div class="f-10 white lh-16" style="font-style:italic;">
                    "{$ai['personal_message']}"
                </div>
            </div>
            <img src="{$logoSrc}" class="logo-sm" style="margin-top:20px; opacity:0.5;" />
        </div>
    </div>

    <!-- SEC 2: HIGHLIGHTS -->
    <div class="sec bg-dark">
        <div class="sec-content">
            <div class="divider divider-white"></div>
            <div class="f-12 fw-bold white mb-15">Destaques deste veículo</div>
            <div>
                {$hlHtml}
            </div>
        </div>
    </div>

    <!-- SEC 3: IMAGE -->
    <div class="sec">
        <div class="sec-img"><img src="{$img2}" /></div>
        <div class="sec-overlay"></div>
        <div class="sec-content" style="display:table; width:100%; height:100%;">
            <div style="display:table-cell; vertical-align:bottom; padding-bottom:20px;">
                <div class="f-8 upper ls-2 white muted mb-5">Interior e Detalhes</div>
                <div class="f-14 fw-bold white">{$v['model']}</div>
            </div>
        </div>
    </div>

    <div class="footer-bar"></div>
</div>

<!-- ====================== PAGE 3 ====================== -->
<div class="page">

    <!-- SEC 1: WHY COVEL -->
    <div class="sec bg-white">
        <div class="sec-content">
            <div class="divider"></div>
            <div class="f-14 fw-bold mb-10">Por que escolher a Covel?</div>
            <div class="f-9 gray lh-16 justify mb-15">
                {$ai['why_choose']}
            </div>
            <div class="f-8 gray lh-16">
                ✓ Mais de 30 anos de experiência &nbsp;&nbsp;
                ✓ Inspeção de 100+ itens &nbsp;&nbsp;
                ✓ Garantia estendida &nbsp;&nbsp;
                ✓ Financiamento facilitado
            </div>
        </div>
    </div>

    <!-- SEC 2: CTA -->
    <div class="sec">
        <div class="sec-img"><img src="{$img3}" /></div>
        <div class="sec-overlay"></div>
        <div class="sec-content center" style="display:table; width:100%; height:100%;">
            <div style="display:table-cell; vertical-align:middle;">
                <div class="f-16 fw-bold white mb-10">{$ai['cta']}</div>
                <div class="f-9 white muted">
                    📍 Rua Julio de Castilhos, 1795 - Encantado/RS<br>
                    📞 (51) 3751-1234 &nbsp;|&nbsp; 💬 WhatsApp
                </div>
            </div>
        </div>
    </div>

    <!-- SEC 3: FOOTER -->
    <div class="sec bg-off">
        <div class="sec-content center" style="display:table; width:100%; height:100%;">
            <div style="display:table-cell; vertical-align:middle;">
                <img src="{$logoSrc}" class="logo mb-10" />
                <div class="f-8 gray ls-1 upper">Covel Veículos</div>
                <div class="f-8 gray muted" style="margin-top:15px;">
                    Este catálogo foi gerado exclusivamente para {$userName}<br>
                    Válido por 7 dias a partir da data de emissão
                </div>
            </div>
        </div>
    </div>

    <div class="footer-bar"></div>
</div>

</body>
</html>
HTML;

// Generate PDF
$options = new Options();
$options->set('isHtml5ParserEnabled', true);
$options->set('isRemoteEnabled', true);
$options->set('defaultFont', 'Helvetica');

$dompdf = new Dompdf($options);
$dompdf->loadHtml($html);
$dompdf->setPaper('A4', 'portrait');

try {
    $dompdf->render();
    $dompdf->stream("catalogo_{$vehicleId}.pdf", ["Attachment" => true]);
} catch (\Exception $e) {
    http_response_code(500);
    echo "Error: " . $e->getMessage();
}
