/**
 * Covel PDF Catalog Generator - VERSÃO FINAL DE PRODUÇÃO
 * Implementa Polling Assíncrono para resiliência contra instabilidades de rede e timeouts.
 */
(function () {
    console.log("%c 🚀 Covel PDF Generator v7 (Resiliência Total)", "color: #0044cc; font-weight: bold; font-size: 16px;");

    // ================= CONFIGURAÇÃO =================
    // ⚠️ SUBSTITUA PELA SUA URL DE PRODUÇÃO (ex: https://pdf.covelrs.com.br)
    const BASE_URL = 'https://SUA-URL-DE-PRODUCAO.com';

    const API_ENDPOINT = BASE_URL + '/generate_lp.php';
    const CHECK_ENDPOINT = BASE_URL + '/check_pdf.php';
    const DOWNLOAD_ENDPOINT = BASE_URL + '/download_pdf.php';
    // ================================================

    /**
     * Extrai o ID do veículo da URL ou DOM
     */
    function getVehicleId() {
        // Estratégia 1: URL padrão do site
        const urlMatch = window.location.href.match(/-(\d+)\.html/);
        if (urlMatch && urlMatch[1]) return urlMatch[1];

        // Estratégia 2: Iframe do Credere
        const iframe = document.querySelector('#credere-pnp-iframe');
        if (iframe) {
            try {
                const parts = new URL(iframe.src).pathname.split('/');
                const idx = parts.indexOf('veiculo');
                if (idx > -1) return parts[idx + 1];
            } catch (e) { }
        }

        return prompt("Digite o ID do veículo para continuar:", "");
    }

    /**
     * Cria e exibe o modal de personalização
     */
    function createModal(onSubmit) {
        const existing = document.getElementById('covel-pdf-modal-overlay');
        if (existing) existing.remove();

        const modalHtml = `
            <div id="covel-pdf-modal-overlay" style="position: fixed; inset: 0; background: rgba(0,0,0,0.85); z-index: 999999; display: flex; align-items: center; justify-content: center; backdrop-filter: blur(5px); font-family: 'Segoe UI', Roboto, Helvetica, Arial, sans-serif;">
                <div style="background: white; padding: 40px; border-radius: 20px; width: 90%; max-width: 420px; text-align: center; box-shadow: 0 25px 70px rgba(0,0,0,0.5); position: relative; overflow: hidden;">
                    <div style="position: absolute; top: 0; left: 0; right: 0; height: 6px; background: linear-gradient(90deg, #0044cc, #00c6ff);"></div>
                    <div style="font-size: 60px; margin-bottom: 20px;">📄</div>
                    <h2 style="margin: 0 0 10px; color: #111; font-size: 24px;">Catálogo Premium</h2>
                    <p style="color: #666; margin-bottom: 25px; font-size: 15px; line-height: 1.5;">Prepare um catálogo exclusivo com IA para o seu cliente:</p>
                    
                    <div style="text-align: left; margin-bottom: 25px;">
                        <label style="display: block; font-size: 12px; text-transform: uppercase; color: #999; font-weight: bold; margin-bottom: 8px; margin-left: 2px;">Nome do Cliente</label>
                        <input type="text" id="covel-user-name" placeholder="Ex: João Silva" style="width: 100%; padding: 14px; border: 2px solid #eee; border-radius: 12px; font-size: 16px; box-sizing: border-box; outline: none; transition: border-color 0.3s ease;">
                    </div>

                    <button id="covel-modal-submit" style="width: 100%; padding: 16px; background: linear-gradient(135deg, #0044cc, #00c6ff); color: white; border: none; border-radius: 12px; font-weight: bold; cursor: pointer; font-size: 16px; box-shadow: 0 8px 20px rgba(0,68,204,0.3); transition: transform 0.2s ease;">✨ Gerar com IA e Baixar</button>
                    
                    <button id="covel-modal-close" style="margin-top: 15px; background: transparent; border: none; color: #aaa; cursor: pointer; font-size: 14px; text-decoration: underline;">Cancelar</button>
                </div>
            </div>
        `;

        const div = document.createElement('div');
        div.innerHTML = modalHtml;
        document.body.appendChild(div.firstElementChild);

        const input = document.getElementById('covel-user-name');
        input.focus();

        // Estilo focus inline
        input.onfocus = () => { input.style.borderColor = '#0044cc'; };
        input.onblur = () => { input.style.borderColor = '#eee'; };

        document.getElementById('covel-modal-submit').onclick = () => {
            const name = input.value.trim() || "Cliente VIP";
            document.getElementById('covel-pdf-modal-overlay').remove();
            onSubmit(name);
        };

        document.getElementById('covel-modal-close').onclick = () => {
            document.getElementById('covel-pdf-modal-overlay').remove();
        };
    }

    /**
     * Solicita geração e monitora progresso (Polling)
     */
    async function requestPDF(vehicleId, userName, btn) {
        const uniqueId = `covel_${vehicleId}_${Math.random().toString(36).substring(7)}`;
        const originalText = btn.innerHTML;
        btn.innerHTML = "⏳ Iniciando tarefa...";
        btn.disabled = true;

        console.log(`📡 Gerando ID: ${uniqueId}`);

        // Dispara geração (Fire and Forget)
        fetch(API_ENDPOINT, {
            method: 'POST',
            mode: 'cors',
            body: JSON.stringify({ id: vehicleId, user_name: userName, uniqueId: uniqueId })
        }).catch(() => {
            // Alguns timeouts de túnel caem aqui, mas o polling resolve
        });

        let attempts = 0;
        const maxAttempts = 90; // 3 minutos

        const checkInterval = setInterval(async () => {
            attempts++;
            btn.innerHTML = `⏳ IA Pensando (${attempts}s)...`;

            try {
                const response = await fetch(`${CHECK_ENDPOINT}?fileId=${uniqueId}`);
                if (!response.ok) return;

                const data = await response.json();

                if (data.ready) {
                    clearInterval(checkInterval);
                    console.log("✅ PDF pronto! Iniciando download...");
                    btn.innerHTML = "✅ Quase lá...";

                    // Download resiliente via Blob
                    const fileResponse = await fetch(`${DOWNLOAD_ENDPOINT}?fileId=${uniqueId}`);
                    const blob = await fileResponse.blob();
                    const url = window.URL.createObjectURL(blob);

                    const a = document.createElement('a');
                    a.style.display = 'none';
                    a.href = url;
                    a.download = `Catalogo_Covel_${vehicleId}.pdf`;
                    document.body.appendChild(a);
                    a.click();

                    setTimeout(() => {
                        window.URL.revokeObjectURL(url);
                        a.remove();
                    }, 2000);

                    btn.innerHTML = originalText;
                    btn.disabled = false;
                    console.log("🚀 Catálogo entregue com sucesso.");
                }
            } catch (err) { /* Silent retry */ }

            if (attempts >= maxAttempts) {
                clearInterval(checkInterval);
                alert("O servidor demorou muito para responder. Por favor, tente novamente em alguns instantes.");
                btn.innerHTML = originalText;
                btn.disabled = false;
            }
        }, 2000);
    }

    /**
     * Injeção do botão de ação
     */
    function injectButton() {
        const target = document.getElementById('credere-pnp');
        const btn = document.createElement('button');
        btn.id = 'covel-ia-generator-btn';
        btn.innerHTML = "📄 Baixar Catálogo Premium (IA)";

        // Estilo Premium
        const style = "width: 100%; padding: 18px; margin-bottom: 20px; background: linear-gradient(135deg, #0044cc 0%, #00c6ff 100%); color: white; border: none; border-radius: 50px; font-weight: bold; cursor: pointer; text-transform: uppercase; letter-spacing: 1px; box-shadow: 0 8px 25px rgba(0,68,204,0.3); font-family: sans-serif; transition: all 0.3s ease;";
        btn.style.cssText = style;

        btn.onmouseover = () => { btn.style.transform = "translateY(-2px)"; btn.style.boxShadow = "0 10px 30px rgba(0,68,204,0.4)"; };
        btn.onmouseout = () => { btn.style.transform = "none"; btn.style.boxShadow = "0 8px 25px rgba(0,68,204,0.3)"; };

        btn.onclick = (e) => {
            e.preventDefault();
            e.stopPropagation();
            const vid = getVehicleId();
            if (vid) createModal((name) => requestPDF(vid, name, btn));
        };

        if (target) {
            target.parentNode.insertBefore(btn, target);
        } else {
            // Fallback: Botão flutuante se não achar o local padrão
            btn.style.position = "fixed";
            btn.style.bottom = "30px";
            btn.style.right = "30px";
            btn.style.width = "auto";
            btn.style.zIndex = "999999";
            btn.style.padding = "16px 30px";
            document.body.appendChild(btn);
        }
    }

    // Inicialização
    if (document.readyState === 'complete') {
        injectButton();
    } else {
        window.addEventListener('load', injectButton);
    }

})();
