<?php
/**
 * AI Service - Abstraction layer for AI text generation
 * Supports: Ollama (local), OpenAI, Anthropic
 */

class AIService {
    
    // Configuration - Change this to switch providers
    private const PROVIDER = 'ollama'; // 'ollama', 'openai', 'anthropic'
    
    // Ollama settings
    private const OLLAMA_URL = 'http://host.docker.internal:11434/api/generate';
    private const OLLAMA_MODEL = 'llama3.2:1b';
    
    // OpenAI settings (for future use)
    private const OPENAI_URL = 'https://api.openai.com/v1/chat/completions';
    private const OPENAI_MODEL = 'gpt-4o-mini';
    private const OPENAI_KEY = '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';
    
    // Anthropic settings (for future use)
    private const ANTHROPIC_URL = 'https://api.anthropic.com/v1/messages';
    private const ANTHROPIC_MODEL = 'claude-3-haiku-20240307';
    private const ANTHROPIC_KEY = '';
    
    /**
     * Generate ALL content for the vehicle catalog
     */
    public static function generateCatalogContent(array $vehicleData): array {
        $content = [
            'headline' => '',
            'tagline' => '',
            'description' => '',
            'highlights' => [],
            'why_choose' => '',
            'cta' => '',
            'personal_message' => ''
        ];
        
        try {
            // Generate all content in one prompt for efficiency
            $prompt = self::buildCatalogPrompt($vehicleData);
            $response = self::generate($prompt);
            
            if ($response) {
                $content = self::parseCatalogResponse($response, $vehicleData);
            }
        } catch (Exception $e) {
            error_log("AI Catalog Error: " . $e->getMessage());
        }
        
        // Fill with fallbacks if needed
        return self::ensureFallbacks($content, $vehicleData);
    }

    /**
     * Generate content STRICTLY based on provided data (New LP Mode)
     */
    public static function generateStrictCatalogContent(array $vehicleData): array {
        $content = [
            'headline' => '',
            'tagline' => '',
            'sales_copy' => '', // Texto unificado de venda
            'highlights_list' => [], // Lista de bullets reais
            'cta' => ''
        ];
        
        try {
            $prompt = self::buildStrictCatalogPrompt($vehicleData);
            $response = self::generate($prompt);
            
            if ($response) {
                $content = self::parseStrictResponse($response);
            }
        } catch (Exception $e) {
            error_log("AI Strict Catalog Error: " . $e->getMessage());
        }
        
        return self::ensureStrictFallbacks($content, $vehicleData);
    }
    
    /**
     * Build comprehensive catalog prompt
     */
    private static function buildCatalogPrompt(array $data): string {
        $year = $data['year'] ?? ($data['year_man'] ?? '') . '/' . ($data['year_mod'] ?? '');
        return "Você é um copywriter especializado em marketing automotivo premium. Crie conteúdo para um catálogo de seminovo.

DADOS DO VEÍCULO:
- Modelo: {$data['model']}
- Ano: {$year}
- Quilometragem: {$data['km']}
- Combustível: {$data['fuel']}
- Câmbio: {$data['gear']}
- Cor: {$data['color']}
- Acessórios: {$data['accessories']}

GERE O SEGUINTE CONTEÚDO (responda EXATAMENTE neste formato):

HEADLINE: [Frase impactante de máximo 6 palavras que capture a essência do carro]

TAGLINE: [Subtítulo elegante de 10-15 palavras]

DESCRIÇÃO: [2-3 parágrafos persuasivos sobre o veículo, destacando qualidade, conforto e tecnologia. Não invente especificações.]

DESTAQUE1: [Primeiro ponto forte em uma frase curta]
DESTAQUE2: [Segundo ponto forte em uma frase curta]
DESTAQUE3: [Terceiro ponto forte em uma frase curta]
DESTAQUE4: [Quarto ponto forte em uma frase curta]

PORQUE: [Um parágrafo sobre por que escolher a Covel Veículos - procedência, garantia, atendimento]

CTA: [Chamada para ação convincente em uma frase]

MENSAGEM: [Uma mensagem pessoal de 2 frases para o cliente, agradecendo o interesse e convidando para conhecer o veículo]

REGRAS:
- Escreva em português brasileiro
- Tom profissional e sofisticado
- Não mencione preço
- Não invente especificações técnicas";
    }

    /**
     * Build STRICT prompt for LP
     */
    private static function buildStrictCatalogPrompt(array $data): string {
        $year = $data['year'] ?? '';
        $acc = empty($data['accessories']) ? 'Itens de série' : $data['accessories'];
        $userName = $data['user_name'] ?? 'Cliente VIP';
        
        return "ATUE COMO UM ESPECIALISTA EM VENDAS DE CARROS PREMIUM. SEU OBJETIVO É CRIAR CONTEÚDO PARA UM CATÁLOGO EXCLUSIVO DA COVEL VEÍCULOS.
        
IMPORTANTE: ESTE CATÁLOGO É PERSONALIZADO PARA O CLIENTE: {$userName}. 

DADOS DO VEÍCULO (A VERDADE ABSOLUTA):
- Modelo: {$data['model']}
- Ano: {$year}
- KM: {$data['km']}
- Câmbio: {$data['gear']}
- Combustível: {$data['fuel']}
- Cor: {$data['color']}
- Lista de Opcionais/Acessórios: {$acc}

INSTRUÇÕES DE ESCRITA:
1. O texto deve ser EXCLUSIVO para o cliente {$userName}.
2. Use um tom de consultoria e exclusividade.
3. Não invente nenhuma característica que não esteja na lista.
4. O 'sales_copy' deve ser persuasivo, mencionando como o carro se encaixa no perfil do cliente e citando acessórios reais da lista.

ESTRUTURA DA RESPOSTA (Retorne APENAS o JSON válido, sem markdown):

{
    \"headline\": \"Uma frase curta (max 7 palavras) impactante\",
    \"tagline\": \"Uma sub-headline persuasiva dirigida ao cliente\",
    \"welcome_message\": \"Uma mensagem de boas-vindas calorosa e exclusiva de 1 parágrafo para o cliente {$userName}, agradecendo a preferência e reforçando a exclusividade deste catálogo.\",
    \"sales_copy\": \"Um texto de venda de 2 parágrafos altamente personalizado para {$userName}. Use gatilhos mentais de oportunidade. Cite acessórios reais listados para provar a veracidade.\",
    \"highlights_list\": [
        \"Destaque 1 real\",
        \"Destaque 2 real\",
        \"Destaque 3 real\",
        \"Destaque 4 real\"
    ],
    \"cta\": \"Uma chamada para ação direta para o WhatsApp\"
}";
    }
    
    /**
     * Parse AI response into structured content
     */
    private static function parseCatalogResponse(string $response, array $data): array {
        $content = [];
        
        // Extract each section using regex
        if (preg_match('/HEADLINE:\s*(.+?)(?=TAGLINE:|$)/si', $response, $m)) {
            $content['headline'] = trim($m[1]);
        }
        
        if (preg_match('/TAGLINE:\s*(.+?)(?=DESCRIÇÃO:|DESCRICAO:|$)/si', $response, $m)) {
            $content['tagline'] = trim($m[1]);
        }
        
        if (preg_match('/DESCRI[ÇC][ÃA]O:\s*(.+?)(?=DESTAQUE1:|$)/si', $response, $m)) {
            $content['description'] = trim($m[1]);
        }
        
        // Extract highlights
        $highlights = [];
        for ($i = 1; $i <= 4; $i++) {
            if (preg_match("/DESTAQUE{$i}:\s*(.+?)(?=DESTAQUE|PORQUE:|$)/si", $response, $m)) {
                $highlights[] = trim($m[1]);
            }
        }
        $content['highlights'] = $highlights;
        
        if (preg_match('/PORQUE:\s*(.+?)(?=CTA:|$)/si', $response, $m)) {
            $content['why_choose'] = trim($m[1]);
        }
        
        if (preg_match('/CTA:\s*(.+?)(?=MENSAGEM:|$)/si', $response, $m)) {
            $content['cta'] = trim($m[1]);
        }
        
        if (preg_match('/MENSAGEM:\s*(.+?)(?=REGRAS:|$)/si', $response, $m)) {
            $content['personal_message'] = trim($m[1]);
        }
        
        return $content;
    }

    private static function parseStrictResponse(string $response, array $data = []): array {
        // Tenta limpar markdown de código se houver
        $clean = preg_replace('/^```json\s*|\s*```$/', '', trim($response));
        $json = json_decode($clean, true);
        
        if (json_last_error() === JSON_ERROR_NONE && is_array($json)) {
            return $json;
        }
        
        // Fallback se o JSON falhar
        return [];
    }

    private static function ensureStrictFallbacks(array $content, array $data): array {
        $userName = $data['user_name'] ?? 'Cliente VIP';
        if (empty($content['headline'])) $content['headline'] = $data['model'];
        if (empty($content['tagline'])) $content['tagline'] = "Uma oportunidade única Covel Veículos";
        if (empty($content['welcome_message'])) $content['welcome_message'] = "Prezado(a) {$userName}, é um prazer apresentar este catálogo exclusivo preparado especialmente para você. Na Covel Veículos, priorizamos a transparência e a qualidade em cada detalhe.";
        if (empty($content['sales_copy'])) $content['sales_copy'] = "Confira este excelente {$data['model']}. Veículo periciado e com garantia.";
        if (empty($content['highlights_list'])) $content['highlights_list'] = ["Seminovo Certificado", "Garantia de Procedência", "Inspeção Cautelar Aprovada"];
        if (empty($content['cta'])) $content['cta'] = "Fale Agora no WhatsApp";
        
        return $content;
    }
    
    /**
     * Ensure all fields have content (fallbacks)
     */
    private static function ensureFallbacks(array $content, array $data): array {
        $modelName = explode(' ', $data['model'])[0] ?? 'Veículo';
        
        if (empty($content['headline'])) {
            $content['headline'] = "Elegância e Performance";
        }
        
        if (empty($content['tagline'])) {
            $content['tagline'] = "Descubra um novo padrão de dirigir com o {$data['model']}";
        }
        
        if (empty($content['description'])) {
            $year = $data['year'] ?? ($data['year_mod'] ?? '');
            $content['description'] = "O {$data['model']} {$year} representa o equilíbrio perfeito entre sofisticação e praticidade. Com apenas {$data['km']} rodados, este seminovo certificado oferece toda a confiabilidade que você procura.\n\nEquipado com câmbio {$data['gear']} e motor {$data['fuel']}, proporciona uma experiência de condução suave e econômica. Cada detalhe foi pensado para seu conforto.";
        }
        
        if (empty($content['highlights']) || count($content['highlights']) < 3) {
            $content['highlights'] = [
                "Baixa quilometragem: {$data['km']}",
                "Manutenção em dia e revisado",
                "Procedência garantida Covel",
                "Financiamento facilitado"
            ];
        }
        
        if (empty($content['why_choose'])) {
            $content['why_choose'] = "Na Covel Veículos, cada seminovo passa por rigorosa inspeção de mais de 100 itens. Oferecemos garantia estendida, financiamento facilitado e a tranquilidade de comprar de quem entende de carros há mais de 30 anos.";
        }
        
        if (empty($content['cta'])) {
            $content['cta'] = "Agende seu test-drive e apaixone-se!";
        }
        
        if (empty($content['personal_message'])) {
            $content['personal_message'] = "Obrigado pelo seu interesse neste veículo! Estamos à disposição para tirar suas dúvidas e agendar uma visita. Será um prazer atendê-lo!";
        }
        
        return $content;
    }
    
    /**
     * Main generation method - routes to appropriate provider
     */
    private static function generate(string $prompt): ?string {
        switch (self::PROVIDER) {
            case 'ollama':
                return self::generateOllama($prompt);
            case 'openai':
                return self::generateOpenAI($prompt);
            case 'anthropic':
                return self::generateAnthropic($prompt);
            default:
                throw new Exception("Unknown AI provider: " . self::PROVIDER);
        }
    }
    
    /**
     * Ollama (Local) generation
     */
    private static function generateOllama(string $prompt): ?string {
        $data = [
            'model' => self::OLLAMA_MODEL,
            'prompt' => $prompt,
            'stream' => false,
            'options' => [
                'temperature' => 0.7,
                'num_predict' => 800
            ]
        ];
        
        $ch = curl_init(self::OLLAMA_URL);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 120
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            throw new Exception("Ollama error: HTTP $httpCode");
        }
        
        $result = json_decode($response, true);
        return $result['response'] ?? null;
    }
    
    /**
     * OpenAI generation
     */
    private static function generateOpenAI(string $prompt): ?string {
        if (empty(self::OPENAI_KEY)) {
            throw new Exception("OpenAI API key not configured");
        }
        
        $data = [
            'model' => self::OPENAI_MODEL,
            'messages' => [
                ['role' => 'user', 'content' => $prompt]
            ],
            'max_tokens' => 800,
            'temperature' => 0.7
        ];
        
        $ch = curl_init(self::OPENAI_URL);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . self::OPENAI_KEY
            ],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 60
        ]);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        $result = json_decode($response, true);
        return $result['choices'][0]['message']['content'] ?? null;
    }
    
    /**
     * Anthropic (Claude) generation
     */
    private static function generateAnthropic(string $prompt): ?string {
        if (empty(self::ANTHROPIC_KEY)) {
            throw new Exception("Anthropic API key not configured");
        }
        
        $data = [
            'model' => self::ANTHROPIC_MODEL,
            'max_tokens' => 800,
            'messages' => [
                ['role' => 'user', 'content' => $prompt]
            ]
        ];
        
        $ch = curl_init(self::ANTHROPIC_URL);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'x-api-key: ' . self::ANTHROPIC_KEY,
                'anthropic-version: 2023-06-01'
            ],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 60
        ]);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        $result = json_decode($response, true);
        return $result['content'][0]['text'] ?? null;
    }
}
